import React from 'react';
import {
  ScrollView,
  StatusBar,
  StyleSheet,
  Text,
  TextInput,
  View,
  Keyboard,
  Alert,
  TouchableOpacity,
  Linking,
  ActivityIndicator,
  Switch,
  Platform,
} from 'react-native';
import {SafeAreaProvider, SafeAreaView} from 'react-native-safe-area-context';
import Result from './Result';
import {createOrder} from './Api';
const TrustyAppScheme = 'jl-app-uat://customer';
function App() {
  const [name, setName] = React.useState('Your product name');
  const [amount, setAmount] = React.useState('1000');
  const [loading, setLoading] = React.useState(false);
  const [resultVisible, setResultVisible] = React.useState(false);
  const [qrcodePay, setQrcodePay] = React.useState(false);
  const [qrcodeLink, setQrcodeLink] = React.useState('');
  const [orderId, setOrderId] = React.useState<number>();

  function onChangeAmount(text: string) {
    if (text.match(/^\d*$/)) {
      setAmount(text);
    }
  }
  async function onSubmit() {
    if (loading) {
      return;
    }
    Keyboard.dismiss();
    if (!name) {
      Alert.alert('Please input product name');
      return;
    }
    if (!amount) {
      Alert.alert('Please input payment amount');
      return;
    }
    setLoading(true);
    if (!qrcodePay) {
      let canOpen = await Linking.canOpenURL(TrustyAppScheme);
      if (!canOpen) {
        setLoading(false);
        Alert.alert(
          "You don't have Trusty Pay(UAT) installed on your phone",
          '',
          [
            {
              text: 'Cancel',
              style: 'cancel',
            },
            {
              text: 'Download',
              onPress: () => {
                Linking.openURL(
                  Platform.select({
                    ios: 'https://www.pgyer.com/smMO',
                    android: 'https://www.pgyer.com/3nzE',
                  })!,
                );
              },
            },
          ],
        );
        return;
      }
    }
    let res = await createOrder({
      amount: Number(amount),
      name: name,
    });
    setLoading(false);
    console.log(res.data);

    if (res.data?.code === 200) {
      setOrderId(res.data.data.id);
      setResultVisible(true);
      let appUrl = res.data.data.appUrl;
      if (qrcodePay) {
        setQrcodeLink(appUrl);
      } else {
        Linking.openURL(appUrl);
      }
    } else {
      Alert.alert(res.data?.msg || 'Request Error: ' + res.status);
    }
  }

  return (
    <SafeAreaProvider>
      <StatusBar
        barStyle={'dark-content'}
        backgroundColor={styles.container.backgroundColor}
        translucent={true}
      />
      <View style={styles.container}>
        <ScrollView
          contentInsetAdjustmentBehavior="automatic"
          style={styles.scrollview}>
          <SafeAreaView mode="padding" style={styles.scrollviewContent}>
            <View style={styles.sectionContainer}>
              <Text style={styles.sectionTitle}>Trusty Pay Demo</Text>
              <Text style={styles.sectionDescription}>
                Fill in the form to create an order
              </Text>
            </View>
            <View style={styles.formItem}>
              <Text style={styles.label}>Product Name</Text>
              <View style={styles.inputWrap}>
                <TextInput
                  style={styles.input}
                  value={name}
                  onChangeText={setName}
                  placeholderTextColor="#ccc"
                  placeholder="Please input product name"
                />
              </View>
            </View>
            <View style={styles.formItem}>
              <Text style={styles.label}>Payment Amount</Text>
              <View style={styles.inputWrap}>
                <TextInput
                  style={styles.input}
                  value={amount}
                  onChangeText={onChangeAmount}
                  placeholderTextColor="#ccc"
                  placeholder="Please input payment amount"
                  keyboardType="number-pad"
                />
              </View>
            </View>
            <View style={[styles.formItem, styles.formRowItem]}>
              <Text style={styles.label}>QR code payment</Text>
              <Switch value={qrcodePay} onValueChange={setQrcodePay} />
            </View>
            <TouchableOpacity
              disabled={loading}
              style={styles.buttonWrap}
              onPress={onSubmit}>
              {loading ? <ActivityIndicator /> : null}
              <Text style={styles.buttonText}>Submit</Text>
            </TouchableOpacity>
          </SafeAreaView>
        </ScrollView>
        <Result
          orderId={orderId}
          qrcode={qrcodeLink}
          visible={resultVisible}
          onClose={() => {
            setLoading(false);
            setOrderId(undefined);
            setQrcodeLink('');
            setResultVisible(false);
          }}
        />
      </View>
    </SafeAreaProvider>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollview: {
    backgroundColor: '#fff',
  },
  scrollviewContent: {
    paddingHorizontal: 24,
  },
  sectionContainer: {
    marginTop: 32,
    marginBottom: 80,
  },
  sectionTitle: {
    fontSize: 24,
    color: '#000',
    fontWeight: '600',
  },
  sectionDescription: {
    marginTop: 8,
    fontSize: 14,
    color: '#888',
  },
  formItem: {
    marginBottom: 24,
    borderBottomWidth: StyleSheet.hairlineWidth,
    borderBottomColor: '#f5f5f5',
  },
  formRowItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  label: {
    fontSize: 14,
    color: '#333',
  },
  inputWrap: {
    paddingVertical: 6,
    paddingHorizontal: 12,
    marginTop: 8,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 2,
  },
  input: {
    padding: 0,
    flex: 1,
    fontSize: 16,
    color: '#000',
  },
  buttonWrap: {
    marginTop: 30,
    backgroundColor: '#4d03d4',
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    height: 40,
    borderRadius: 4,
  },
  buttonText: {
    marginLeft: 4,
    fontSize: 16,
    color: '#fff',
  },
});

export default App;
