import {
  StyleSheet,
  View,
  Text,
  TextStyle,
  TouchableOpacity,
  ScrollView,
  ActivityIndicator,
  AppState,
  Animated,
  ViewProps,
  StatusBar,
  Alert,
  Platform,
} from 'react-native';
import React from 'react';
import dayjs from 'dayjs';
import QRCode from 'react-native-qrcode-svg';
import {useSafeAreaInsets} from 'react-native-safe-area-context';
import {queryOrder, OrderDetailResult, OrderStatus} from './Api';
type ResultProps = {
  visible: boolean;
  orderId?: number;
  qrcode?: string;
  onClose?: () => void;
};
function FadeInView(props: ViewProps) {
  const fadeAnim = React.useRef(new Animated.Value(0)).current;
  const insets = useSafeAreaInsets();
  const top = Platform.OS === 'ios' ? 0 : insets.top;
  React.useEffect(() => {
    Animated.timing(fadeAnim, {
      toValue: 1,
      duration: 200,
      useNativeDriver: true,
    }).start(); // 开始执行动画
  }, [fadeAnim]);

  return (
    <Animated.View
      style={[
        styles.container,
        {
          top,
          opacity: fadeAnim,
        },
      ]}>
      <StatusBar
        barStyle={'dark-content'}
        backgroundColor={styles.container.backgroundColor}
        translucent={true}
      />
      {props.children}
    </Animated.View>
  );
}
export default function Result(props: ResultProps) {
  const [showQr, setShowQr] = React.useState(false);
  React.useEffect(() => {
    if (props.visible && props.qrcode) {
      setShowQr(true);
    } else {
      setShowQr(false);
    }
  }, [props.visible, props.qrcode]);

  if (!props.visible || !props.orderId) {
    return null;
  }
  if (showQr) {
    return (
      <QrcodeView
        orderId={props.orderId}
        qrcode={props.qrcode}
        onClose={props.onClose}
      />
    );
  }
  return <AppPaymentView orderId={props.orderId} onClose={props.onClose} />;
}

function AppPaymentView(props: {
  orderId: ResultProps['orderId'];
  onClose: ResultProps['onClose'];
}) {
  const appState = React.useRef(AppState.currentState);
  const [orderDetail, setOrderDetail] = React.useState<OrderDetailResult>();
  const [loading, setLoading] = React.useState(true);

  const fetchDetail = React.useCallback(async (orderId: number) => {
    console.log(orderId, 'detail');
    setLoading(true);
    let res = await queryOrder(orderId);
    setLoading(false);
    console.log(res.data);
    if (res.data?.code === 200) {
      setOrderDetail(res.data.data);
      setLoading(false);
    } else {
      Alert.alert(res.data?.msg || 'Request Error: ' + res.status);
      setLoading(false);
    }
  }, []);
  React.useEffect(() => {
    let listener = AppState.addEventListener('change', nextAppState => {
      console.log(nextAppState);
      if (
        appState.current.match(/inactive|background/) &&
        nextAppState === 'active' &&
        props.orderId
      ) {
        fetchDetail(props.orderId);
      }
      appState.current = nextAppState;
    });
    return () => {
      listener.remove();
    };
  }, [props.orderId, fetchDetail]);
  return (
    <FadeInView>
      <View style={styles.contentWrap}>
        {loading ? (
          <View style={styles.loading}>
            <ActivityIndicator />
          </View>
        ) : (
          <OrderDetailView data={orderDetail} onClose={props.onClose} />
        )}
      </View>
    </FadeInView>
  );
}

function QrcodeView(props: {
  orderId: NonNullable<ResultProps['orderId']>;
  qrcode: ResultProps['qrcode'];
  onClose: ResultProps['onClose'];
}) {
  const [orderDetail, setOrderDetail] = React.useState<OrderDetailResult>();
  const [showResult, setShowResult] = React.useState(false);
  const interval = 5000;
  const maxTry = 60;
  const timerRef = React.useRef<number>();
  const timeCounter = React.useRef<number>(0);
  const fetchDetail = React.useCallback(async (orderId: number) => {
    timeCounter.current = timeCounter.current + 1;
    let res = await queryOrder(orderId);
    console.log(res.data);
    if (res.data?.code === 200) {
      const data = res.data.data;
      if (
        data.status === OrderStatus.Success ||
        data.status === OrderStatus.Fail ||
        data.status === OrderStatus.Refund
      ) {
        setOrderDetail(res.data.data);
        setShowResult(true);
        return;
      } else if (timeCounter.current >= maxTry) {
        setOrderDetail(res.data.data);
        setShowResult(true);
        return;
      }
    }
    if (timeCounter.current >= maxTry) {
      setShowResult(true);
      return;
    }
    timerRef.current = setTimeout(() => fetchDetail(orderId), interval);
  }, []);

  React.useEffect(() => {
    timerRef.current && clearTimeout(timerRef.current);
    timerRef.current = setTimeout(() => fetchDetail(props.orderId), interval);
    return () => {
      console.log('cleart timer');
      timerRef.current && clearTimeout(timerRef.current);
    };
  }, [fetchDetail, props.orderId]);

  return (
    <FadeInView>
      <View style={styles.contentWrap}>
        {showResult ? (
          <OrderDetailView data={orderDetail} onClose={props.onClose} />
        ) : (
          <>
            <View style={styles.qrcode}>
              {props.qrcode ? (
                <QRCode value={props.qrcode} size={200} />
              ) : (
                <Text>Error: qrcode is empty</Text>
              )}
              <Text style={styles.qrcodeText}>
                Using Trusty Pay to scan the QR code
              </Text>
            </View>
            <TouchableOpacity style={styles.button} onPress={props.onClose}>
              <Text style={styles.buttonText}>Close</Text>
            </TouchableOpacity>
          </>
        )}
      </View>
    </FadeInView>
  );
}

function OrderDetailView(props: {
  data?: OrderDetailResult;
  onClose: ResultProps['onClose'];
}) {
  const data = props.data;
  const statusText = React.useMemo<{text?: string; style?: TextStyle}>(() => {
    switch (data?.status) {
      case OrderStatus.Processing:
        return {
          style: {color: 'red'},
          text: 'Unpaid',
        };
      case OrderStatus.Success:
        return {
          style: {color: 'green', fontWeight: 'bold'},
          text: 'Success',
        };
      case OrderStatus.Fail:
        return {
          style: {color: 'red', fontWeight: 'bold'},
          text: 'Failure',
        };
      case OrderStatus.Refund:
        return {
          text: 'Refund',
        };
      default:
        return {
          text: data?.status,
        };
    }
  }, [data]);
  const createTime = React.useMemo(() => {
    if (data?.createTime) {
      return dayjs(data.createTime).format('YYYY-MM-DD HH:mm:ss');
    }
    return '';
  }, [data]);
  return (
    <View style={styles.detailContainer}>
      <Text style={styles.title}>Payment Result</Text>
      <ScrollView>
        <View style={styles.cell}>
          <Text style={styles.label}>Order Id:</Text>
          <Text style={styles.text}>{data?.id}</Text>
        </View>
        <View style={styles.cell}>
          <Text style={styles.label}>Order Status:</Text>
          <Text style={[styles.text, statusText.style]}>{statusText.text}</Text>
        </View>
        <View style={styles.cell}>
          <Text style={styles.label}>Product Name:</Text>
          <Text style={styles.text} numberOfLines={2}>
            {data?.desc}
          </Text>
        </View>
        <View style={styles.cell}>
          <Text style={styles.label}>Payment Amount:</Text>
          <Text style={styles.text}>{data?.amount} MMK</Text>
        </View>
        <View style={styles.cell}>
          <Text style={styles.label}>Create Time:</Text>
          <Text style={styles.text}>{createTime}</Text>
        </View>
      </ScrollView>
      <TouchableOpacity style={styles.button} onPress={props.onClose}>
        <Text style={styles.buttonText}>Close</Text>
      </TouchableOpacity>
    </View>
  );
}

const styles = StyleSheet.create({
  container: {
    ...StyleSheet.absoluteFillObject,
    backgroundColor: 'rgba(0,0,0,0.4)',
    justifyContent: 'center',
    alignItems: 'stretch',
  },
  contentWrap: {
    marginHorizontal: 24,
    height: 500,
    backgroundColor: '#fff',
    borderRadius: 6,
  },
  detailContainer: {
    flex: 1,
  },
  title: {
    margin: 20,
    fontSize: 24,
    color: '#333',
    fontWeight: '500',
  },
  cell: {
    marginHorizontal: 12,
    padding: 12,
    flexDirection: 'row',
    alignItems: 'center',
    borderBottomColor: '#c5c5c5',
    borderBottomWidth: StyleSheet.hairlineWidth,
  },
  label: {
    width: 130,
    fontSize: 14,
    color: '#888',
  },
  text: {
    flex: 1,
    fontSize: 14,
    color: '#000',
  },
  success: {
    fontWeight: '500',
    color: 'green',
  },
  failure: {
    fontWeight: '500',
    color: 'red',
  },
  button: {
    alignItems: 'center',
    marginHorizontal: 40,
    marginVertical: 20,
    padding: 12,
    borderRadius: 4,
    backgroundColor: '#aaa',
  },
  buttonText: {
    fontSize: 14,
    color: '#fff',
  },
  loading: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  qrcode: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  qrcodeText: {
    marginTop: 20,
    fontSize: 14,
    color: '#333',
  },
});
